USE @UT_DB_NAME@
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[dbo].[utilfn_GetColumnListForInsert]') and xtype in (N'FN', N'IF', N'TF'))
drop function [dbo].[utilfn_GetColumnListForInsert]
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS ON 
GO

/*******************************************************************************
**	Name: 	utilfn_GetColumnListForInsert
**
**	Desc:	Returns a comma delimited string of columns for the input table.
**		The intent was to use this to create dynamic INSERT statements.
**		This will bypass Identity columns and RowVersion since they 
**		cannot be directly populated.
**
**	Input Parameters:
**		tableName - Target table to get column list for
**
**	Author:	Lohse
**	Date:	04/01/2011
********************************************************************************
**	Date Modified		Developer		Description
**	-------------		----------		------------------------
**	05/29/2012		Lohse			SQL 2008 AutoVal changed
**	06/22/2012		Lohse			Adding IDENTITYS, doing full insert - screw it!
********************************************************************************/
CREATE   FUNCTION dbo.utilfn_GetColumnListForInsert
(
	@tableName 	SYSNAME
)  
RETURNS VARCHAR(4000) AS  

BEGIN 

	DECLARE @columnsList VARCHAR(4000)

	SELECT 
		@columnsList = COALESCE( @columnsList + ', ', '' ) + SC.[name]
	FROM
		sysobjects SO JOIN syscolumns SC ON SO.id = SC.id
    	WHERE 
		SO.xtype = 'U'
	AND	SO.type = 'U'
	AND	SO.name = @tableName
	-- don't include RowVersion since they are not updatable
	AND	SC.name <> 'RowVersion' 	

	ORDER BY SC.colid

RETURN(@columnsList)

END


GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[dbo].[utilsp_KillDBUsers]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [dbo].[utilsp_KillDBUsers]
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS ON 
GO

/*******************************************************************************
**	Name: 	utilsp_KillDBUsers
**
**	Desc:	Kills all processes attached to input argument database
**
**	Input Parameters:
**		databaseName - Target database to remove active processes for
**
**	Author:	Lohse
**	Date:	04/01/2011
********************************************************************************
**	Date Modified		Developer		Description
**	-------------		----------		------------------------
********************************************************************************/
CREATE PROCEDURE dbo.utilsp_KillDBUsers
(
	@databaseName SYSNAME
)	
AS	
	DECLARE @activeDbUsers TABLE
	( 
		DbUserID 	INT IDENTITY(1,1),
		UserSpid 	SMALLINT NOT NULL 
	)

	INSERT INTO @activeDbUsers
	( 
		UserSpid
	)
	SELECT spid
	FROM
		master..sysprocesses P JOIN master..sysdatabases D ON P.dbid = D.dbid
	WHERE    
		D.[name] = @databaseName
	AND	spid <> @@SPID		--grab all process except THIS process

	--ABORT ON ERRORS******************************************************************
	IF @@ERROR <> 0 
	BEGIN
		RAISERROR( 'Error creating a table of database users', 16, 1 )
		RETURN( 1 )
	END

	DECLARE @sql			NVARCHAR(4000)
	DECLARE	@userRecordId		INT
	DECLARE	@numberOfUsersToKill 	INT
	DECLARE	@errorBuf		INT
	

	SELECT @numberOfUsersToKill = COUNT(1) FROM @activeDbUsers

	--CYCLE THROW ALL PROCESSES AND DYNAMICALLY "KILL" THEN 
	SET @userRecordId = 1
	WHILE @userRecordId <= @numberOfUsersToKill
	BEGIN
		SELECT 
			@sql = N'KILL ' + CAST( UserSpid AS NVARCHAR(30) ) 
		FROM 
			@activeDbUsers 
		WHERE 
			DbUserID = @userRecordId
		
		IF @@ERROR <> 0 
		BEGIN    
			RAISERROR( 'Error preparing dynamic SQL to kill user', 16, 1 )
			RETURN( 1 )
		END
		
		EXEC( @sql )
		
		IF @@ERROR <> 0 
		BEGIN
			RAISERROR( 'Error trying to KILL process with dynamic SQL:%s', 16, 1, @sql )
			RETURN( 1 )
		END                
		
		SET @userRecordId = @userRecordId + 1
	END

RETURN( 0 )

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO


if exists (select * from dbo.sysobjects where id = object_id(N'[dbo].[utilfn_GetMirrorViewName]') and xtype in (N'FN', N'IF', N'TF'))
drop function [dbo].[utilfn_GetMirrorViewName]
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS ON 
GO

/*******************************************************************************
**	Name: 	utilfn_GetMirrorViewName
**
**	Desc:	Returns View name for a Mirror Table.
**		This  just prepends a dbo.V_ to the input argument.
**
**	Input Parameters:
**		tableeName - Target table to generate a Mirror View name for
**
**	Author:	Lohse
**	Date:	04/01/2011
********************************************************************************
**	Date Modified		Developer		Description
**	-------------		----------		------------------------
********************************************************************************/
CREATE  FUNCTION dbo.utilfn_GetMirrorViewName
( 
	@tableName	SYSNAME 
)  
RETURNS SYSNAME AS  
BEGIN 
	RETURN 'dbo.V_' + OBJECT_NAME( OBJECT_ID( @tableName ) ) 
END


GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO


if exists (select * from dbo.sysobjects where id = object_id(N'[dbo].[utilfn_GetMirrorTriggerName]') and xtype in (N'FN', N'IF', N'TF'))
drop function [dbo].[utilfn_GetMirrorTriggerName]
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS ON 
GO

/*******************************************************************************
**	Name: 	utilfn_GetMirrorViewName
**
**	Desc:	Returns View name for a Mirror Table.
**		This  just prepends a dbo.V_ to the input argument.
**
**	Input Parameters:
**		tableeName - Target table to generate a Mirror View name for
**
**	Author:	Lohse
**	Date:	04/01/2011
********************************************************************************
**	Date Modified		Developer		Description
**	-------------		----------		------------------------
********************************************************************************/
CREATE  FUNCTION dbo.utilfn_GetMirrorTriggerName 
(
	@tableName	SYSNAME
)  
RETURNS SYSNAME AS  

BEGIN 
	RETURN 'dbo.trMirror' + OBJECT_NAME( OBJECT_ID( @tableName ) ) 
END


GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO


if exists (select * from dbo.sysobjects where id = object_id(N'[dbo].[utilfn_GetMirrorTableName]') and xtype in (N'FN', N'IF', N'TF'))
drop function [dbo].[utilfn_GetMirrorTableName]
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS ON 
GO

/*******************************************************************************
**	Name: 	utilfn_GetMirrorTableName
**
**	Desc:	Returns table name - stripped - no dbo, etc. 
**
**	Input Parameters:
**		tableeName - Target table to get the name
**
**	Author:	Lohse
**	Date:	04/01/2011
********************************************************************************
**	Date Modified		Developer		Description
**	-------------		----------		------------------------
********************************************************************************/
CREATE FUNCTION dbo.utilfn_GetMirrorTableName
(
	@tableName	SYSNAME
)  

RETURNS SYSNAME AS  

BEGIN 
	RETURN OBJECT_NAME( OBJECT_ID( @tableName ) )  -- striping 'dbo' and such
END

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO


if exists (select * from dbo.sysobjects where id = object_id(N'[dbo].[utilfn_GetMirrorDbName]') and xtype in (N'FN', N'IF', N'TF'))
drop function [dbo].[utilfn_GetMirrorDbName]
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS ON 
GO
/*******************************************************************************
**	Name: 	utilfn_GetMirrorDbName
**
**	Desc:	Returns Mirror Database name for current database.
**		Basically, this appends _MIRROR to DB_NAME()
**
**	Input Parameters:
**		None.
**
**	Author:	Lohse
**	Date:	04/01/2011
********************************************************************************
**	Date Modified		Developer		Description
**	-------------		----------		------------------------
********************************************************************************/
CREATE FUNCTION dbo.utilfn_GetMirrorDbName()  
RETURNS SYSNAME AS  

BEGIN 
	RETURN DB_NAME() + N'_MIRROR'
END

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[dbo].[utilfn_GetDbFilesPath]') and xtype in (N'FN', N'IF', N'TF'))
drop function [dbo].[utilfn_GetDbFilesPath]
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS ON 
GO

/*******************************************************************************
**	Name: 	utilfn_GetDbFilesPath
**
**	Desc:	Returns fully qualified path of MDF file for the current database
**		e.g. C:\Program Files\Microsoft SQL Server\MSSQL\data\
**
**	Input Parameters:
**		None.
**
**	Author:	Lohse
**	Date:	04/01/2011
********************************************************************************
**	Date Modified		Developer		Description
**	-------------		----------		------------------------
********************************************************************************/
CREATE FUNCTION dbo.utilfn_GetDbFilesPath()
RETURNS NVARCHAR(4000) AS

BEGIN
	DECLARE @targetDbFilesPath NVARCHAR(4000)
	
	SELECT 
		@targetDbFilesPath = SUBSTRING( fileName, 1, CHARINDEX( DB_NAME(), fileName ) - 1 ) 
	FROM 
		master.dbo.sysdatabases 
	WHERE 
		[name] = DB_NAME()
	
	RETURN @targetDbFilesPath
END


GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[dbo].[utilfn_ReplaceRegularExpression]') and xtype in (N'FN', N'IF', N'TF'))
drop function [dbo].[utilfn_ReplaceRegularExpression]
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS ON 
GO

/*******************************************************************************
**	Name: 	utilfn_ReplaceRegularExpression
**
**	Desc:	Uses a regular expression to replace string text.
**		Returns the new string with the replacments.
**
**	Input Parameters:
**		source		source string that needs replacing
**		regexp 		regular expression to match the string against
**		replacestr	string to replace any matches with
**		casesensitive	string comparison is case sensitive
**
**	Author:	Lohse
**	Date:	04/01/2011
********************************************************************************
**	Date Modified		Developer		Description
**	-------------		----------		------------------------
********************************************************************************/
CREATE  FUNCTION dbo.utilfn_ReplaceRegularExpression 
(
    @source        VARCHAR(8000),
    @regexp        VARCHAR(500),
    @replacestr    VARCHAR(8000),
    @casesensitive BIT
)
RETURNS VARCHAR(8000) AS 
BEGIN
	DECLARE @hr 		INTEGER
	DECLARE @objHandle 	INTEGER
	DECLARE @result 	VARCHAR(8000)

	SET @casesensitive = (-@casesensitive*2+2)/2	--no idea why this is here

	EXEC @hr = sp_OACreate 'VBScript.RegExp', @objHandle OUTPUT
	EXEC @hr = sp_OASetProperty @objHandle, 'Pattern', @regexp
	EXEC @hr = sp_OASetProperty @objHandle, 'Global', 'true'
	EXEC @hr = sp_OASetProperty @objHandle, 'IgnoreCase', @casesensitive
	EXEC @hr = sp_OAMethod @objHandle, 'Replace', @result OUTPUT, @source, @replacestr
	EXEC @hr = sp_OADestroy @objHandle

	RETURN @result
END



GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO



if exists (select * from dbo.sysobjects where id = object_id(N'[dbo].[utilsp_UpdateMirrorDbNameInRoutines]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [dbo].[utilsp_UpdateMirrorDbNameInRoutines]
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS ON 
GO


/*******************************************************************************
**	Name: 	utilsp_UpdateMirrorDbNameInRoutines
**
**	Desc:	Replaces all occurrances of "VBECS_MIRROR" in routines that
**		start with (sp, ba, fn).
**		Changes VBECS_MIRROR to the actual Mirror database name
**
**	Input Parameters:
**		None.
**
**	Author:	Lohse
**	Date:	04/01/2011
********************************************************************************
**	Date Modified		Developer		Description
**	-------------		----------		------------------------
********************************************************************************/
CREATE PROCEDURE dbo.utilsp_UpdateMirrorDbNameInRoutines
AS

SET NOCOUNT, ANSI_NULLS ON

DECLARE	@retVal	INT 
SET @retVal = 1 	-- fail by default

DECLARE @RoutinesToProcess TABLE
( 
	RoutineId	INT IDENTITY(1,1),
	RoutineName	SYSNAME,
	RoutineTypeName SYSNAME
)

IF @@ERROR <> 0 
BEGIN
	RAISERROR( 'Error creating temporary table @RoutinesToProcess', 16, 1 )
	SET @retVal = 1
	GOTO CLEANUP_AND_EXIT
END


INSERT INTO 
	@RoutinesToProcess( RoutineName, RoutineTypeName )
SELECT DISTINCT
	O.[name],
	CASE LEFT( O.[name], 2 ) 
	WHEN 'sp' THEN 'PROCEDURE'
	WHEN 'ba' THEN 'PROCEDURE'
	WHEN 'fn' THEN 'FUNCTION'
	END 
FROM 
	SYSOBJECTS O JOIN SYSCOMMENTS C ON O.ID = C.ID
WHERE 
	O.xtype IN ('P','IF', 'TF', 'FN')
AND	(O.[name] LIKE 'sp%' OR O.[name] LIKE 'batchsp%' OR O.[name] LIKE 'fn%')
AND	C.TEXT LIKE '%@DB_NAME@_MIRROR.%'
ORDER BY
	O.NAME 

IF @@ERROR <> 0 
BEGIN
	RAISERROR( 'Error inserting routines to process into @RoutinesToProcess', 16, 1 )
	SET @retVal = 1
	GOTO CLEANUP_AND_EXIT
END

CREATE TABLE #originalRoutineText
(        
LineId INT IDENTITY(1,1),
TextLine VARCHAR(1024)
)

IF @@ERROR <> 0 
BEGIN
	RAISERROR( 'Error creating temporary table #originalRoutineText', 16, 1 )
	SET @retVal = 1
	GOTO CLEANUP_AND_EXIT
END

CREATE TABLE #updatedRoutineText
(        
LineId INT IDENTITY(1,1),
TextLine VARCHAR(1024)
)

IF @@ERROR <> 0 
BEGIN
	RAISERROR( 'Error creating temporary table #updatedRoutineText', 16, 1 )
	SET @retVal = 1
	GOTO CLEANUP_AND_EXIT
END

DECLARE	@currentRoutineId 	INT
DECLARE	@numberOfRoutinesToProcess 	INT
DECLARE	@routineTypeName 	SYSNAME
DECLARE	@routineName 		SYSNAME
DECLARE	@currentLineNumber 	INT
DECLARE	@numberOfLinesInRoutine INT
DECLARE	@line 			VARCHAR(1024)
DECLARE	@errorBuf 		INT

-- Variables needed to fit sproc text in a single EXEC statement (MS recommended way)
DECLARE	@sqlstr1	VARCHAR(4000)
DECLARE	@sqlstr2	VARCHAR(4000)
DECLARE	@sqlstr3	VARCHAR(4000)
DECLARE	@sqlstr4	VARCHAR(4000)
DECLARE	@sqlstr5	VARCHAR(4000)
DECLARE	@sqlstr6	VARCHAR(4000)
DECLARE	@sqlstr7	VARCHAR(4000)
DECLARE	@sqlstr8	VARCHAR(4000)
DECLARE	@sqlstr9	VARCHAR(4000)
DECLARE	@sqlstr10	VARCHAR(4000)
DECLARE	@sqlstr11	VARCHAR(4000)
DECLARE	@sqlstr12	VARCHAR(4000)
DECLARE	@sqlstr13	VARCHAR(4000)
DECLARE	@sqlstr14	VARCHAR(4000)
DECLARE	@sqlstr15	VARCHAR(4000)

SET @currentRoutineId = 1
SELECT @numberOfRoutinesToProcess = COUNT(1) FROM @RoutinesToProcess

IF @@ERROR <> 0 
BEGIN
	RAISERROR( 'Error setting the variable @numberOfRoutinesToProcess', 16, 1 )
	SET @retVal = 1
	GOTO CLEANUP_AND_EXIT
END

WHILE @currentRoutineId <= @numberOfRoutinesToProcess
BEGIN
	SELECT
		@routineName = RoutineName,
		@routineTypeName = RoutineTypeName
	FROM
		@RoutinesToProcess
	WHERE
		RoutineId = @currentRoutineId

	IF @@ERROR <> 0 
	BEGIN
		RAISERROR( 'Error populating variables from @RoutinesToProcess', 16, 1 )
		GOTO CLEANUP_AND_EXIT
	END

	INSERT INTO #originalRoutineText
	(
		TextLine
	) 
	EXEC @errorBuf = sp_helptext @routineName

	IF COALESCE(@errorBuf,@@ERROR ) <> 0
	BEGIN
		RAISERROR( 'Error fetching routine text into #originalRoutineText', 16, 1 )
		SET @retVal = 1
		GOTO CLEANUP_AND_EXIT
	END


	INSERT INTO #updatedRoutineText
	(
		TextLine
	)
	SELECT 
		dbo.utilfn_ReplaceRegularExpression( 
			REPLACE( TextLine, '@DB_NAME@_MIRROR.', dbo.utilfn_GetMirrorDbName() + '.' ), 
			'CREATE\s{1,80}' + @routineTypeName, 
			'ALTER ' + @routineTypeName, 
			1 )
	FROM 
		#originalRoutineText 
	ORDER BY LineId

	IF @@ERROR <> 0 
	BEGIN
		RAISERROR( 'Error inserting replaced text into #updatedRoutineText', 16, 1 )
		SET @retVal = 1
		GOTO CLEANUP_AND_EXIT
	END

	SELECT 
		@sqlstr1 = '',
		@sqlstr2 = '',
		@sqlstr3 = '',
		@sqlstr4 = '',
		@sqlstr5 = '',
		@sqlstr6 = '',
		@sqlstr7 = '',
		@sqlstr8 = '',
		@sqlstr9 = '',
		@sqlstr10 = '',
		@sqlstr11 = '',
		@sqlstr12 = '',
		@sqlstr13 = '',
		@sqlstr14 = '',
		@sqlstr15 = ''

	SET @currentLineNumber = 1

	SELECT @numberOfLinesInRoutine = COUNT(1) FROM #updatedRoutineText

	IF @@ERROR <> 0 
	BEGIN
		RAISERROR( 'Error setting variable @numberOfLinesInRoutine', 16, 1 )
		SET @retVal = 1
		GOTO CLEANUP_AND_EXIT
	END


	WHILE @currentLineNumber <= @numberOfLinesInRoutine
	BEGIN
		SELECT 
			@line = TextLine 
		FROM 
			#updatedRoutineText 
		WHERE 
			LineId = @currentLineNumber

	IF @@ERROR <> 0 
	BEGIN
		RAISERROR( 'Error fetching text from #updatedRoutineText', 16, 1 )
		SET @retVal = 1
		GOTO CLEANUP_AND_EXIT
	END

	IF LEN(@sqlstr1) < 2976 SET @sqlstr1 = @sqlstr1 + @line
	ELSE
	IF LEN(@sqlstr2) < 2976 SET @sqlstr2 = @sqlstr2 + @line
	ELSE
	IF LEN(@sqlstr3) < 2976 SET @sqlstr3 = @sqlstr3 + @line
	ELSE
	IF LEN(@sqlstr4) < 2976 SET @sqlstr4 = @sqlstr4 + @line
	ELSE
	IF LEN(@sqlstr5) < 2976 SET @sqlstr5 = @sqlstr5 + @line
	ELSE
	IF LEN(@sqlstr6) < 2976 SET @sqlstr6 = @sqlstr6 + @line
	ELSE
	IF LEN(@sqlstr7) < 2976 SET @sqlstr7 = @sqlstr7 + @line
	ELSE
	IF LEN(@sqlstr8) < 2976 SET @sqlstr8 = @sqlstr8 + @line
	ELSE
	IF LEN(@sqlstr9) < 2976 SET @sqlstr9 = @sqlstr9 + @line
	ELSE
	IF LEN(@sqlstr10) < 2976 SET @sqlstr10 = @sqlstr10 + @line
	ELSE
	IF LEN(@sqlstr11) < 2976 SET @sqlstr11 = @sqlstr11 + @line
	ELSE
	IF LEN(@sqlstr12) < 2976 SET @sqlstr12 = @sqlstr12 + @line
	ELSE
	IF LEN(@sqlstr13) < 2976 SET @sqlstr13 = @sqlstr13 + @line
	ELSE
	IF LEN(@sqlstr14) < 2976 SET @sqlstr14 = @sqlstr14 + @line
	ELSE
	IF LEN(@sqlstr15) < 2976 SET @sqlstr15 = @sqlstr15 + @line
	
	IF @@ERROR <> 0 
	BEGIN
		RAISERROR( 'Error splitting text to into string variables', 16, 1 )
		SET @retVal = 1
		GOTO CLEANUP_AND_EXIT
	END

	SET @currentLineNumber = @currentLineNumber + 1
END 

EXEC( @sqlstr1 + @sqlstr2 + @sqlstr3 + @sqlstr4 + @sqlstr5 + @sqlstr6 + @sqlstr7 + @sqlstr8 + @sqlstr9 + @sqlstr10 + @sqlstr11 + @sqlstr12 + @sqlstr13 + @sqlstr14 + @sqlstr15 )

IF @@ERROR <> 0 
BEGIN
	RAISERROR( 'Error executing updated routine text: %s', 16, 1, @routineName )
	SET @retVal = 1
	GOTO CLEANUP_AND_EXIT
END

TRUNCATE TABLE #originalRoutineText

IF @@ERROR <> 0
BEGIN
	RAISERROR( 'Error truncating temporary table #originalRoutineText', 16, 1 )
	SET @retVal = 1
	GOTO CLEANUP_AND_EXIT
END

TRUNCATE TABLE #updatedRoutineText    

IF @@ERROR <> 0
BEGIN
	RAISERROR( 'Error truncating temporary table #updatedRoutineText', 16, 1 )
	SET @retVal = 1
	GOTO CLEANUP_AND_EXIT
END

SET @currentRoutineId = @currentRoutineId + 1
END

SET @retVal = 0

CLEANUP_AND_EXIT:
SET NOCOUNT OFF

IF OBJECT_ID( 'tempdb..#originalRoutineText' ) IS NOT NULL 
BEGIN
	DROP TABLE #originalRoutineText

	IF @@ERROR <> 0 
	BEGIN
		RAISERROR( 'Error dropping temporary table #originalRoutineText', 16, 1 )
		SET @retVal = 1 
	END
END

IF OBJECT_ID( 'tempdb..#updatedRoutineText' ) IS NOT NULL
BEGIN
DROP TABLE #updatedRoutineText

IF @@ERROR <> 0 
BEGIN
	RAISERROR( 'Error dropping temporary table #updatedRoutineText', 16, 1 )
	SET @retVal = 1
END
END

    RETURN @retVal
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO



if exists (select * from dbo.sysobjects where id = object_id(N'[dbo].[utilsp_SetNewMirrorDbOptions]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [dbo].[utilsp_SetNewMirrorDbOptions]
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS ON 
GO

/*******************************************************************************************************************************************************************************
**
**	Desc:	Called from utilsp_CreateEmptyMirrorDb, this proc sets various options on the Mirror Database
**
**	Author: 	S. Antropov
**
**	Date: 	09/28/2005
**
*******************************************************************************************************************************************************************************
**	Modified:
**	Date		Name		Description
**	-------------		----------		------------------------------------------------------------------------------------------------------------------------------------------------------------------------
**	10/11/2005	N.Paterson	** Reviewed for compliance with database standards and recommendations **
					Added comment header
					Replaced invalid COALESCE( NULLIF( @@ERROR, 0 ), @errorBuf ) with COALESCE(@errorBuf,@@ERROR)
*******************************************************************************************************************************************************************************/
CREATE PROCEDURE dbo.utilsp_SetNewMirrorDbOptions
AS
    DECLARE @errorBuf INT
    DECLARE @mirrorDbName SYSNAME
    DECLARE @sql NVARCHAR(4000)
    DECLARE @dbOption NVARCHAR(4000)
    DECLARE @optionList NVARCHAR(4000)

    SET @sql = 'ALTER DATABASE ' + @mirrorDbName 
    SET @mirrorDbName = dbo.utilfn_GetMirrorDbName()

    IF @@ERROR <> 0 
    BEGIN
        RAISERROR( 'Error occurred while trying to retrieve VBECS mirror DB name', 16, 1 )
        RETURN( 1 )
    END            

    SET @sql = 'ALTER DATABASE ' + @mirrorDbName + ' SET COMPATIBILITY_LEVEL = 100'
    EXEC @errorBuf = sp_executesql @sql
    IF COALESCE(@errorBuf,@@ERROR) <> 0 
      BEGIN
          RAISERROR( 'Error occurred while trying to set mirror DB COMPATIBILITY_LEVEL option', 16, 1 )
          RETURN( 1 )
      END  

    SET @sql = 'ALTER DATABASE ' + @mirrorDbName + ' SET ANSI_NULL_DEFAULT OFF'
    EXEC @errorBuf = sp_executesql @sql
    IF COALESCE(@errorBuf,@@ERROR) <> 0 
      BEGIN
          RAISERROR( 'Error occurred while trying to set mirror DB ANSI_NULL_DEFAULT option', 16, 1 )
          RETURN( 1 )
      END  


    SET @sql = 'ALTER DATABASE ' + @mirrorDbName + ' SET ANSI_NULLS OFF'
    EXEC @errorBuf = sp_executesql @sql
    IF COALESCE(@errorBuf,@@ERROR) <> 0 
      BEGIN
          RAISERROR( 'Error occurred while trying to set mirror DB ANSI_NULLS option', 16, 1 )
          RETURN( 1 )
      END  


    SET @sql = 'ALTER DATABASE ' + @mirrorDbName + ' SET ANSI_PADDING OFF'
    EXEC @errorBuf = sp_executesql @sql
    IF COALESCE(@errorBuf,@@ERROR) <> 0 
      BEGIN
          RAISERROR( 'Error occurred while trying to set mirror DB ANSI_PADDING option', 16, 1 )
          RETURN( 1 )
      END  

    SET @sql = 'ALTER DATABASE ' + @mirrorDbName + ' SET ANSI_WARNINGS OFF'
    EXEC @errorBuf = sp_executesql @sql
    IF COALESCE(@errorBuf,@@ERROR) <> 0 
      BEGIN
          RAISERROR( 'Error occurred while trying to set mirror DB ANSI_WARNINGS option', 16, 1 )
          RETURN( 1 )
      END  

    SET @sql = 'ALTER DATABASE ' + @mirrorDbName + ' SET ARITHABORT OFF'
    EXEC @errorBuf = sp_executesql @sql
    IF COALESCE(@errorBuf,@@ERROR) <> 0 
      BEGIN
          RAISERROR( 'Error occurred while trying to set mirror DB ARITHABORT option', 16, 1 )
          RETURN( 1 )
      END  

    SET @sql = 'ALTER DATABASE ' + @mirrorDbName + ' SET AUTO_CLOSE OFF'
    EXEC @errorBuf = sp_executesql @sql
    IF COALESCE(@errorBuf,@@ERROR) <> 0 
      BEGIN
          RAISERROR( 'Error occurred while trying to set mirror DB AUTO_CLOSE option', 16, 1 )
          RETURN( 1 )
      END  

    SET @sql = 'ALTER DATABASE ' + @mirrorDbName + ' SET AUTO_CREATE_STATISTICS ON'
    EXEC @errorBuf = sp_executesql @sql
    IF COALESCE(@errorBuf,@@ERROR) <> 0 
      BEGIN
          RAISERROR( 'Error occurred while trying to set mirror DB AUTO_CREATE_STATISTICS option', 16, 1 )
          RETURN( 1 )
      END  

    SET @sql = 'ALTER DATABASE ' + @mirrorDbName + ' SET AUTO_SHRINK OFF'
    EXEC @errorBuf = sp_executesql @sql
    IF COALESCE(@errorBuf,@@ERROR) <> 0 
      BEGIN
          RAISERROR( 'Error occurred while trying to set mirror DB AUTO_SHRINK option', 16, 1 )
          RETURN( 1 )
      END  

    SET @sql = 'ALTER DATABASE ' + @mirrorDbName + ' SET AUTO_UPDATE_STATISTICS ON'
    EXEC @errorBuf = sp_executesql @sql
    IF COALESCE(@errorBuf,@@ERROR) <> 0 
      BEGIN
          RAISERROR( 'Error occurred while trying to set mirror DB AUTO_UPDATE_STATISTICS option', 16, 1 )
          RETURN( 1 )
      END  

    SET @sql = 'ALTER DATABASE ' + @mirrorDbName + ' SET CURSOR_CLOSE_ON_COMMIT OFF'
    EXEC @errorBuf = sp_executesql @sql
    IF COALESCE(@errorBuf,@@ERROR) <> 0 
      BEGIN
          RAISERROR( 'Error occurred while trying to set mirror DB CURSOR_CLOSE_ON_COMMIT option', 16, 1 )
          RETURN( 1 )
      END  

    SET @sql = 'ALTER DATABASE ' + @mirrorDbName + ' SET CURSOR_DEFAULT GLOBAL'
    EXEC @errorBuf = sp_executesql @sql
    IF COALESCE(@errorBuf,@@ERROR) <> 0 
      BEGIN
          RAISERROR( 'Error occurred while trying to set mirror DB CURSOR_DEFAULT option', 16, 1 )
          RETURN( 1 )
      END  

    SET @sql = 'ALTER DATABASE ' + @mirrorDbName + ' SET CONCAT_NULL_YIELDS_NULL OFF'
    EXEC @errorBuf = sp_executesql @sql
    IF COALESCE(@errorBuf,@@ERROR) <> 0 
      BEGIN
          RAISERROR( 'Error occurred while trying to set mirror DB CONCAT_NULL_YIELDS_NULL option', 16, 1 )
          RETURN( 1 )
      END  

    SET @sql = 'ALTER DATABASE ' + @mirrorDbName + ' SET NUMERIC_ROUNDABORT OFF'
    EXEC @errorBuf = sp_executesql @sql
    IF COALESCE(@errorBuf,@@ERROR) <> 0 
      BEGIN
          RAISERROR( 'Error occurred while trying to set mirror DB NUMERIC_ROUNDABORT option', 16, 1 )
          RETURN( 1 )
      END  

    SET @sql = 'ALTER DATABASE ' + @mirrorDbName + ' SET QUOTED_IDENTIFIER OFF'
    EXEC @errorBuf = sp_executesql @sql
    IF COALESCE(@errorBuf,@@ERROR) <> 0 
      BEGIN
          RAISERROR( 'Error occurred while trying to set mirror DB QUOTED_IDENTIFIER option', 16, 1 )
          RETURN( 1 )
      END  

    SET @sql = 'ALTER DATABASE ' + @mirrorDbName + ' SET RECURSIVE_TRIGGERS OFF'
    EXEC @errorBuf = sp_executesql @sql
    IF COALESCE(@errorBuf,@@ERROR) <> 0 
      BEGIN
          RAISERROR( 'Error occurred while trying to set mirror DB RECURSIVE_TRIGGERS option', 16, 1 )
          RETURN( 1 )
      END  

    SET @sql = 'ALTER DATABASE ' + @mirrorDbName + ' SET DISABLE_BROKER'
    EXEC @errorBuf = sp_executesql @sql
    IF COALESCE(@errorBuf,@@ERROR) <> 0 
      BEGIN
          RAISERROR( 'Error occurred while trying to set mirror DB DISABLE_BROKER option', 16, 1 )
          RETURN( 1 )
      END  

    SET @sql = 'ALTER DATABASE ' + @mirrorDbName + ' SET AUTO_UPDATE_STATISTICS_ASYNC OFF'
    EXEC @errorBuf = sp_executesql @sql
    IF COALESCE(@errorBuf,@@ERROR) <> 0 
      BEGIN
          RAISERROR( 'Error occurred while trying to set mirror DB AUTO_UPDATE_STATISTICS_ASYNC option', 16, 1 )
          RETURN( 1 )
      END  


    SET @sql = 'ALTER DATABASE ' + @mirrorDbName + ' SET DATE_CORRELATION_OPTIMIZATION OFF'
    EXEC @errorBuf = sp_executesql @sql
    IF COALESCE(@errorBuf,@@ERROR) <> 0 
      BEGIN
          RAISERROR( 'Error occurred while trying to set mirror DB DATE_CORRELATION_OPTIMIZATION option', 16, 1 )
          RETURN( 1 )
      END  

    SET @sql = 'ALTER DATABASE ' + @mirrorDbName + ' SET TRUSTWORTHY OFF'
    EXEC @errorBuf = sp_executesql @sql
    IF COALESCE(@errorBuf,@@ERROR) <> 0 
      BEGIN
          RAISERROR( 'Error occurred while trying to set mirror DB TRUSTWORTHY option', 16, 1 )
          RETURN( 1 )
      END  

    SET @sql = 'ALTER DATABASE ' + @mirrorDbName + ' SET ALLOW_SNAPSHOT_ISOLATION OFF'
    EXEC @errorBuf = sp_executesql @sql
    IF COALESCE(@errorBuf,@@ERROR) <> 0 
      BEGIN
          RAISERROR( 'Error occurred while trying to set mirror DB ALLOW_SNAPSHOT_ISOLATION option', 16, 1 )
          RETURN( 1 )
      END  

    SET @sql = 'ALTER DATABASE ' + @mirrorDbName + ' SET PARAMETERIZATION SIMPLE'
    EXEC @errorBuf = sp_executesql @sql
    IF COALESCE(@errorBuf,@@ERROR) <> 0 
      BEGIN
          RAISERROR( 'Error occurred while trying to set mirror DB PARAMETERIZATION option', 16, 1 )
          RETURN( 1 )
      END  

    SET @sql = 'ALTER DATABASE ' + @mirrorDbName + ' SET READ_COMMITTED_SNAPSHOT OFF'
    EXEC @errorBuf = sp_executesql @sql
    IF COALESCE(@errorBuf,@@ERROR) <> 0 
      BEGIN
          RAISERROR( 'Error occurred while trying to set mirror DB READ_COMMITTED_SNAPSHOT option', 16, 1 )
          RETURN( 1 )
      END  

    SET @sql = 'ALTER DATABASE ' + @mirrorDbName + ' SET HONOR_BROKER_PRIORITY OFF'
    EXEC @errorBuf = sp_executesql @sql
    IF COALESCE(@errorBuf,@@ERROR) <> 0 
      BEGIN
          RAISERROR( 'Error occurred while trying to set mirror DB HONOR_BROKER_PRIORITY option', 16, 1 )
          RETURN( 1 )
      END  

    SET @sql = 'ALTER DATABASE ' + @mirrorDbName + ' SET READ_WRITE'
    EXEC @errorBuf = sp_executesql @sql
    IF COALESCE(@errorBuf,@@ERROR) <> 0 
      BEGIN
          RAISERROR( 'Error occurred while trying to set mirror DB READ_WRITE option', 16, 1 )
          RETURN( 1 )
      END  

    SET @sql = 'ALTER DATABASE ' + @mirrorDbName + ' SET RECOVERY SIMPLE'
    EXEC @errorBuf = sp_executesql @sql
    IF COALESCE(@errorBuf,@@ERROR) <> 0 
      BEGIN
          RAISERROR( 'Error occurred while trying to set mirror DB RECOVERY option', 16, 1 )
          RETURN( 1 )
      END  

    SET @sql = 'ALTER DATABASE ' + @mirrorDbName + ' SET MULTI_USER '
    EXEC @errorBuf = sp_executesql @sql
    IF COALESCE(@errorBuf,@@ERROR) <> 0 
      BEGIN
          RAISERROR( 'Error occurred while trying to set mirror DB MULTI_USER option', 16, 1 )
          RETURN( 1 )
      END  

    SET @sql = 'ALTER DATABASE ' + @mirrorDbName + ' SET PAGE_VERIFY TORN_PAGE_DETECTION '
    EXEC @errorBuf = sp_executesql @sql
    IF COALESCE(@errorBuf,@@ERROR) <> 0 
      BEGIN
          RAISERROR( 'Error occurred while trying to set mirror DB PAGE_VERIFY TORN_PAGE_DETECTION option', 16, 1 )
          RETURN( 1 )
      END  

    SET @sql = 'ALTER DATABASE ' + @mirrorDbName + ' SET DB_CHAINING OFF'
    EXEC @errorBuf = sp_executesql @sql
    IF COALESCE(@errorBuf,@@ERROR) <> 0 
      BEGIN
          RAISERROR( 'Error occurred while trying to set mirror DB DB_CHAINING OFF option', 16, 1 )
          RETURN( 1 )
      END  




    RETURN( 0 )
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO



if exists (select * from dbo.sysobjects where id = object_id(N'[dbo].[utilsp_EnableAllMirrorTriggers]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [dbo].[utilsp_EnableAllMirrorTriggers]
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS ON 
GO

/*******************************************************************************
**	Name: 	utilsp_EnableAllMirrorTriggers
**
**	Desc:	Enables or Diables all triggers that start with trMirror
**
**	Input Parameters:
**		enable - true to enable (default), false to disable
**
**	Author:	Lohse
**	Date:	04/01/2011
********************************************************************************
**	Date Modified		Developer		Description
**	-------------		----------		------------------------
********************************************************************************/
CREATE  PROCEDURE dbo.utilsp_EnableAllMirrorTriggers
(
	@enable bit = 1
)
AS
	DECLARE @TriggersToProcess TABLE
	(
		TriggerId	INT IDENTITY(1,1),
		TriggerName	SYSNAME
	)

	INSERT INTO @TriggersToProcess
	SELECT 
		name
	FROM
		sysobjects 
	WHERE
		name LIKE 'trMirror%' 
	AND	xtype = 'TR'

	IF @@ERROR <> 0 
	BEGIN
	    RAISERROR( 'Error selecting list of mirror triggers into @TriggersToProcess', 16, 1 )
	    RETURN( 1 )
	END


	DECLARE	@option		NVARCHAR(10)
	DECLARE @numberOfTriggers INT
	DECLARE @TriggerId 	INT
	DECLARE @sql 		NVARCHAR(4000)
	DECLARE @errorBuf 	INT

	IF @enable = 1 
		SET @option = N'ENABLE'
	ELSE
		SET @option = N'DISABLE'

	SELECT @numberOfTriggers = COUNT(1) FROM @TriggersToProcess
	SET @TriggerId = 1
		
	WHILE @TriggerId <=  @numberOfTriggers
	BEGIN
		SELECT 
			@sql = 'ALTER TABLE ' + REPLACE( TriggerName, 'trMirror', '' ) + N' ' + @option + ' TRIGGER ' + TriggerName
		FROM
			@TriggersToProcess
		WHERE
			TriggerId = @TriggerId
				
	    IF @@ERROR <> 0 
	    BEGIN
	        RAISERROR( 'Error fetching ALTER TABLE dynamic SQL from table variable', 16, 1 )
	        RETURN( 1 )
	    END
	
	    EXEC @errorBuf = sp_executesql @sql
	
	    IF COALESCE( @errorBuf,@@ERROR)  <> 0 
	    BEGIN 
	        RAISERROR( 'Error executing TRIGGER update dynamic SQL: %s', 16, 1, @sql )
	        RETURN( 1 )
	    END
	
	    SET @TriggerId = @TriggerId + 1
	END
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO


if exists (select * from dbo.sysobjects where id = object_id(N'[dbo].[utilsp_EnableAllReferentialConstraints]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [dbo].[utilsp_EnableAllReferentialConstraints]
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS ON 
GO

/*******************************************************************************
**	Name: 	utilsp_EnableAllReferentialConstraints
**
**	Desc:	Enables or Diables all table check constraints
**
**	Input Parameters:
**		enable - true to enable (default), false to disable
**
**	Author:	Lohse
**	Date:	04/01/2011
********************************************************************************
**	Date Modified		Developer		Description
**	-------------		----------		------------------------
********************************************************************************/
CREATE    PROCEDURE dbo.utilsp_EnableAllReferentialConstraints
(
	@enable BIT = 1
)
AS
	DECLARE @ConstraintsToProcess TABLE
	(
		ConstraintId 	INT IDENTITY(1,1),
		TableName	SYSNAME,
		ConstraintName	SYSNAME
	)

	INSERT INTO @ConstraintsToProcess
	( 
		TableName, 
		ConstraintName 
	)
	SELECT	ctu.TABLE_NAME,
		rc.CONSTRAINT_NAME
	FROM 	
		INFORMATION_SCHEMA.REFERENTIAL_CONSTRAINTS rc JOIN INFORMATION_SCHEMA.CONSTRAINT_TABLE_USAGE ctu
	ON	rc.CONSTRAINT_CATALOG = ctu.CONSTRAINT_CATALOG
	AND	rc.CONSTRAINT_SCHEMA = ctu.CONSTRAINT_SCHEMA
	AND	rc.CONSTRAINT_NAME = ctu.CONSTRAINT_NAME

	IF @@ERROR <> 0 
	BEGIN
	    RAISERROR( 'Error fetching a list of constraints into @ConstraintsToProcess', 16, 1 )
	    RETURN( 1 )
	END

	DECLARE	@option		NVARCHAR(10)
	DECLARE @numberOfConstraints INT
	DECLARE @constraintId	INT
	DECLARE @sql		NVARCHAR(4000) 
	DECLARE @errorBuf	INT

	IF @enable = 1 
		SET @option = N'CHECK'
	ELSE
		SET @option = N'NOCHECK'

	SELECT @numberOfConstraints = COUNT(1) FROM @ConstraintsToProcess
	SET @constraintId = 1
		
	WHILE @constraintId <=  @numberOfConstraints
	BEGIN
		SELECT 
			@sql = 'ALTER TABLE ' + TableName + N' ' + @option + ' CONSTRAINT ' + ConstraintName
		FROM
			@ConstraintsToProcess
		WHERE
			ConstraintId = @constraintId
				
	    IF @@ERROR <> 0 
	    BEGIN
	        RAISERROR( 'Error fetching ALTER TABLE dynamic SQL from @ConstraintsToProcess', 16, 1 )
	        RETURN( 1 )
	    END
	
	    EXEC @errorBuf = sp_executesql @sql
	
	    IF COALESCE( @errorBuf,@@ERROR)  <> 0 
	    BEGIN 
	        RAISERROR( 'Error executing CONSTRAINT update dynamic SQL: %s', 16, 1, @sql )
	        RETURN( 1 )
	    END
	
	    SET @constraintId = @constraintId + 1
	END
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO


if exists (select * from dbo.sysobjects where id = object_id(N'[dbo].[utilsp_ClearUnitTestTableData]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [dbo].[utilsp_ClearUnitTestTableData]
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS ON 
GO

/*******************************************************************************
**	Name: 	utilsp_ClearUnitTestTableData
**
**	Desc:	DELETES all table data for all user tables except
**		('dtproperties')
**
**	Input Parameters:
**		None.
**
**	Author:	Lohse
**	Date:	04/01/2011
********************************************************************************
**	Date Modified		Developer		Description
**	-------------		----------		------------------------
**	03/21/2013		Lohse			Quick fix for config files values
********************************************************************************/
CREATE  PROCEDURE dbo.utilsp_ClearUnitTestTableData
AS
    
	DECLARE @TablesToClear TABLE
	(
		TableId		INT IDENTITY(1,1),
		TableName	SYSNAME
	)
    
    
	INSERT INTO @TablesToClear 
		SELECT   [TableName] =  name
		FROM 
			sysobjects 
		WHERE
		  	type = 'U' 
		AND 	name <> 'dtproperties'
		AND	name NOT LIKE 'tmpl%'
		AND 	name <> 'VBECSGlobalConfig'



	IF @@ERROR <> 0 
	BEGIN
		RAISERROR( 'Error fetching a list of tables into @TablesToClear', 16, 1 )
		RETURN( 1 )
	END

    
	DECLARE @errorBuf INT
	EXEC @errorBuf = dbo.utilsp_EnableAllReferentialConstraints 0
	
	IF COALESCE( NULLIF( @@ERROR, 0 ), @errorBuf ) <> 0
	BEGIN
		RAISERROR( 'Error trying to disable all referential constraints', 16, 1 )
		RETURN( 1 )
	END


	EXEC @errorBuf = dbo.utilsp_EnableAllMirrorTriggers 0
	
	IF COALESCE( NULLIF( @@ERROR, 0 ), @errorBuf ) <> 0
	BEGIN
		RAISERROR( 'Error trying to disable all mirror triggers', 16, 1 )
		RETURN( 1 )
	END


	DECLARE @tableId INT, @numberOfTablesToClear INT, @sql NVARCHAR(4000)


	SELECT @numberOfTablesToClear = COUNT(1) FROM @TablesToClear
	SET @tableId = 1


	WHILE @tableId <= @numberOfTablesToClear
	BEGIN
		SELECT 
			@sql = N'DELETE FROM ' + TableName
		FROM
			@TablesToClear
		WHERE
			TableId = @tableId


		IF @@ERROR <> 0 
		BEGIN
			RAISERROR( 'Error fetching table delete dynamic SQL from @TablesToClear', 16, 1 )
			RETURN( 1 )
		END
            
		EXEC @errorBuf = sp_executesql @sql 

		IF COALESCE( NULLIF( @@ERROR, 0 ), @errorBuf ) <> 0
		BEGIN
    			RAISERROR( 'Error trying to delete table data using dynamic SQL: %s', 16, 1, @sql )
			RETURN( 1 )
		END

		SET @tableId = @tableId + 1
	END    


	EXEC @errorBuf = dbo.utilsp_EnableAllReferentialConstraints 1
	
	IF COALESCE( NULLIF( @@ERROR, 0 ), @errorBuf ) <> 0
	BEGIN
		RAISERROR( 'Error trying to enable all referential constraints', 16, 1 )
		RETURN( 1 )
	END

	EXEC @errorBuf = dbo.utilsp_EnableAllMirrorTriggers 1
	
	IF COALESCE( NULLIF( @@ERROR, 0 ), @errorBuf ) <> 0
	BEGIN
		RAISERROR( 'Error trying to enable all mirror triggers', 16, 1 )
		RETURN( 1 )
	END

RETURN( 0 )

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO



if exists (select * from dbo.sysobjects where id = object_id(N'[dbo].[utilsp_DropMirrorSupportTableIfExist]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [dbo].[utilsp_DropMirrorSupportTableIfExist]
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS ON 
GO

/*******************************************************************************
**	Name: 	utilsp_DropMirrorSupportTableIfExist
**
**	Desc:	Drops tables Class and MirrorIndexFieldLookup if they exist
**
**	Input Parameters:
**		None.
**
**	Author:	Lohse
**	Date:	04/01/2011
********************************************************************************
**	Date Modified		Developer		Description
**	-------------		----------		------------------------
********************************************************************************/
CREATE PROCEDURE dbo.utilsp_DropMirrorSupportTableIfExist
AS

    IF OBJECT_ID( '[Class]' ) IS NOT NULL AND OBJECTPROPERTY( OBJECT_ID( '[Class]' ), 'IsTable' ) = 1
    BEGIN
        DROP TABLE [Class]
        
        IF @@ERROR <> 0 
        BEGIN
            RAISERROR( 'Error trying to drop mirror support table [Class]', 16, 1 )
            RETURN( 1 ) 
        END
    END

    IF OBJECT_ID( '[MirrorIndexFieldLookup]' ) IS NOT NULL AND OBJECTPROPERTY( OBJECT_ID( '[MirrorIndexFieldLookup]' ), 'IsTable' ) = 1
    BEGIN
        DROP TABLE [MirrorIndexFieldLookup]
        
        IF @@ERROR <> 0 
        BEGIN
            RAISERROR( 'Error trying to drop mirror support table [MirrorIndexFieldLookup]', 16, 1 )
            RETURN( 1 ) 
        END
    END

    RETURN( 0 )
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[dbo].[utilsp_ExecuteSprocInMirrorDb]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [dbo].[utilsp_ExecuteSprocInMirrorDb]
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS ON 
GO

/*******************************************************************************
**	Name: 	utilsp_ExecuteSprocInMirrorDb
**
**	Desc:	Dynamically executes a stored procedure in the Mirror database
**
**	Input Parameters:
**		sprocExecString stoted procedure to execute (including any parameters)
**		e.g. sp_who2
**
**	Author:	Lohse
**	Date:	04/01/2011
********************************************************************************
**	Date Modified		Developer		Description
**	-------------		----------		------------------------
********************************************************************************/
CREATE PROCEDURE dbo.utilsp_ExecuteSprocInMirrorDb
(
	@sprocExecString NVARCHAR(3900)
)
AS

	DECLARE @sql NVARCHAR(4000);
	
	SELECT @sql = 'USE ' + dbo.utilfn_GetMirrorDbName() + '; EXEC @sprocError = ' + @sprocExecString
	
	IF @@ERROR <> 0 
	BEGIN
		RAISERROR( 'Error trying to retrieve VBECS mirror DB name', 16, 1 )
		RETURN( 1 )
	END            
    
	DECLARE @errorBuf	INT
	DECLARE @sprocError	INT 

	EXEC @errorBuf = sp_executesql @sql, N'@sprocError INT OUTPUT', @sprocError OUTPUT

	IF COALESCE( @sprocError, @errorBuf ,@@ERROR) <> 0
	BEGIN
		RAISERROR( 'Error trying to execute the following dynamic SQL: %s', 16, 1, @sql )
		RETURN( 1 )
	END

RETURN( 0 )

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO


if exists (select * from dbo.sysobjects where id = object_id(N'[dbo].[utilsp_ExecuteSqlInMirrorDb]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [dbo].[utilsp_ExecuteSqlInMirrorDb]
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS ON 
GO

/*******************************************************************************
**	Name: 	utilsp_ExecuteSqlInMirrorDb
**
**	Desc:	Dynamically executes a TSQL in the Mirror database
**
**	Input Parameters:
**		execStr T-SQL to execute
**
**	Author:	Lohse
**	Date:	04/01/2011
********************************************************************************
**	Date Modified		Developer		Description
**	-------------		----------		------------------------
********************************************************************************/
CREATE PROCEDURE dbo.utilsp_ExecuteSqlInMirrorDb
(
	@execStr NVARCHAR(3900)
)
AS

	DECLARE @sql NVARCHAR(4000);
	
	SELECT @sql = 'USE ' + dbo.utilfn_GetMirrorDbName() + '; ' + @execStr
	
	IF @@ERROR <> 0 
	BEGIN
		RAISERROR( 'Error trying to retrieve VBECS mirror DB name', 16, 1 )
		RETURN( 1 )
	END            
	
	EXEC( @sql )
	
	IF @@ERROR <> 0
	BEGIN
		RAISERROR( 'Error trying to execute the following dynamic SQL: %s', 16, 1, @sql )
		RETURN( 1 )
	END
	
RETURN( 0 )
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO



if exists (select * from dbo.sysobjects where id = object_id(N'[dbo].[utilsp_GrantDbAccessToLogin]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [dbo].[utilsp_GrantDbAccessToLogin]
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS ON 
GO


/*******************************************************************************
**	Name: 	utilsp_GrantDbAccessToLogin
**
**	Desc:	Grants DB access to VbecsAppUser and VbecsAppMirrorUser
**
**	Input Parameters:
**		loginName SYSNAME of the user to grant access to
**
**	Author:	Lohse
**	Date:	04/01/2011
********************************************************************************
**	Date Modified		Developer		Description
**	-------------		----------		------------------------
********************************************************************************/
CREATE  PROCEDURE dbo.utilsp_GrantDbAccessToLogin
(
    @loginName SYSNAME
)
AS

	IF LEFT( RTRIM( @loginName ), 1 ) = '[' 
	BEGIN
		RAISERROR( 'Error: Login name must be specified WITHOUT [ brackets ]', 16, 1 )
		RETURN( 1 )
	END

	IF NOT EXISTS( SELECT 1 FROM master..syslogins WHERE [name] = @loginName )
	BEGIN
		RAISERROR( 'Specified login does not exist in the system catalog', 16, 1 )
		RETURN( 1 )        
	END

	DECLARE @errorBuf INT

	IF EXISTS( SELECT 1 FROM sysusers where [name] = @loginName )
	BEGIN
		EXEC @errorBuf = sp_revokedbaccess @loginName
		IF COALESCE( @errorBuf,@@ERROR)  <> 0 
		BEGIN
			RAISERROR( 'Failed to revoke existing DB access for login %s', 16, 1, @loginName )
			RETURN( 1 )
		END
	END

/*
sp_grantdbaccess
Adds a security account in the current database for a Microsoft SQL Server login or
Microsoft Windows NT user or group, and enables it to be granted permissions
to perform activities in the database.
*/

	EXEC @errorBuf = sp_grantdbaccess @loginName


	IF COALESCE( @errorBuf,@@ERROR)  <> 0 
	BEGIN
		RAISERROR( 'Failed to grant DB access for login %s to the DB', 16, 1, @loginName )
		RETURN( 1 )
	END

	EXEC @errorBuf = sp_addrolemember 'VbecsAppUser', @loginName

	IF COALESCE( @errorBuf,@@ERROR)  <> 0 
	BEGIN
		RAISERROR( 'Failed to add user %s to the role', 16, 1, @loginName )
		RETURN( 1 )
	END

	DECLARE @sql NVARCHAR(4000)

	SET @sql = 
		N'IF EXISTS( SELECT 1 FROM sysusers where [name] = ''' + @loginName + ''' )' + CHAR(13) +
		N'BEGIN ' + CHAR(13) +
		N'    DECLARE @errorBuf INT ' + CHAR(13) + 
		N'    EXEC @errorBuf = sp_revokedbaccess ''' + @loginName + ''' ' + CHAR(13) + 
		N'    IF COALESCE( @errorBuf,@@ERROR)  <> 0' + CHAR(13) + 
		N'         RAISERROR( ''Error revoking existing mirror DB access for login '', 16, 1 )' + CHAR(13) +
		N'END'

	EXEC @errorBuf = dbo.utilsp_ExecuteSqlInMirrorDb @sql

	IF COALESCE( @errorBuf,@@ERROR)  <> 0 
	BEGIN
		RAISERROR( 'Error revoking DB access for login %s in the mirror DB', 16, 1, @loginName )
		RETURN( 1 )
	END

	SET @sql = N'sp_grantdbaccess ''' + @loginName + N''''

	EXEC @errorBuf = dbo.utilsp_ExecuteSprocInMirrorDb @sql

	IF COALESCE( @errorBuf,@@ERROR)  <> 0 
	BEGIN
		RAISERROR( 'Error granting DB access for login %s to the mirror DB', 16, 1, @loginName )
		RETURN( 1 )
	END

	SET @sql = N'sp_addrolemember ''VbecsAppMirrorUser'', ''' + @loginName + N''''

	EXEC @errorBuf = dbo.utilsp_ExecuteSprocInMirrorDb @sql

	IF COALESCE( @errorBuf,@@ERROR)  <> 0 
	BEGIN
		RAISERROR( 'Failed to add user %s to the role in the mirror DB', 16, 1, @loginName )
		RETURN( 1 )
	END

RETURN( 0 )
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO



if exists (select * from dbo.sysobjects where id = object_id(N'[dbo].[utilsp_CreateLoginWithVbecsAccess]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [dbo].[utilsp_CreateLoginWithVbecsAccess]
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS ON 
GO


/*******************************************************************************
**	Name: 	utilsp_CreateLoginWithVbecsAccess
**
**	Desc:	Creates and grants DB access to 2 user groups
**
**	Input Parameters:
**		usersGroup  active directory group for "vbecs users group"
**		adminsGroup active directory group for "vbecs admins group"
**
**	Author:	Lohse
**	Date:	04/01/2011
********************************************************************************
**	Date Modified		Developer		Description
**	-------------		----------		------------------------
********************************************************************************/
CREATE PROCEDURE dbo.utilsp_CreateLoginWithVbecsAccess
(
    @usersGroup SYSNAME, @adminsGroup SYSNAME = ''
)
AS
    IF LEFT( RTRIM( @usersGroup ), 1 ) = '['   
    BEGIN
        RAISERROR( '@usersGroup must be specified WITHOUT [ brackets ]', 16, 1 )
        RETURN( 1 )
    END

    IF LEFT( RTRIM( @adminsGroup ), 1 ) = '['   
    BEGIN
        RAISERROR( '@adminsGroup must be specified WITHOUT [ brackets ]', 16, 1 )
        RETURN( 1 )
    END

    DECLARE @errorBuf INT

    IF NOT EXISTS( SELECT 1 FROM master..syslogins WHERE [name] = @usersGroup )
    BEGIN
--sp_grantlogin
--Allows a Microsoft Windows NT user or group account to connect to Microsoft SQL Server using Windows Authentication.

        EXEC @errorBuf = master..sp_grantlogin @usersGroup

        IF COALESCE( @errorBuf,@@ERROR)  <> 0 
        BEGIN
            RAISERROR( 'Failed to add login %s to the SQL Server', 16, 1, @usersGroup )
            RETURN( 1 )
        END
    END

    EXEC @errorBuf = dbo.utilsp_GrantDbAccessToLogin @usersGroup

    IF COALESCE( @errorBuf,@@ERROR)  <> 0 
    BEGIN
        RAISERROR( 'Failed to grant DB access for login %s', 16, 1, @usersGroup )
        RETURN( 1 )
    END

/*============> process for admin account if @adminName was passed in with anything other than a blank string <===================*/
IF DATALENGTH(REPLACE(COALESCE(@adminsGroup,''),' ','')) > 0
BEGIN
	IF NOT EXISTS( SELECT 1 FROM master..syslogins WHERE [name] = @adminsGroup )
	BEGIN
		EXEC @errorBuf = master..sp_grantlogin @adminsGroup

		IF COALESCE( @errorBuf,@@ERROR)  <> 0 
		BEGIN
			RAISERROR( 'Failed to add Admin login %s to the SQL Server', 16, 1, @adminsGroup )
			RETURN( 1 )
		END
	END

	EXEC @errorBuf = dbo.utilsp_GrantDbAccessToLogin @adminsGroup

	IF COALESCE( @errorBuf,@@ERROR)  <> 0 
	BEGIN
		RAISERROR( 'Failed to grant Admin DB access for login %s', 16, 1, @adminsGroup )
		RETURN( 1 )
	END
END

    RETURN( 0 )
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[dbo].[utilsp_CopyTableFromDataDictionary]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [dbo].[utilsp_CopyTableFromDataDictionary]
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

/*******************************************************************************
**	Name: 	utilsp_CopyTableFromDataDictionary
**
**	Desc:	Copies table from DataDictionary to  UNITTEST database
**
**	Input Parameters:
**		tablename SYSNAME of table to copy
**
**	Author:	Lohse
**	Date:	04/01/2011
********************************************************************************
**	Date Modified		Developer		Description
**	-------------		----------		------------------------
********************************************************************************/
CREATE       PROCEDURE dbo.utilsp_CopyTableFromDataDictionary
(
	@tableName SYSNAME
)
AS

	IF NOT( IS_MEMBER( 'db_owner' ) = 1 )
	BEGIN
		RAISERROR( 'Current user is not a member of db_owner group', 16, 1 )
		RETURN( 1 )
	END
	
	IF @tableName IS NULL OR LEN( @tableName ) = 0
	BEGIN
		RAISERROR( 'Table name parameter is not supplied or NULL', 16, 1 )
		RETURN( 1 )
	END
	
	DECLARE	@adjustedTableName SYSNAME,
			@sql NVARCHAR(4000), 
			@errorBuf INT
	
	SET @adjustedTableName = LTRIM( RTRIM( @tableName ) )
	
	IF LEFT( @adjustedTableName, 4 ) <> 'dbo.' AND LEFT( @adjustedTableName, 6 ) <> '[dbo].'
		SET @adjustedTableName = '[dbo].' + QUOTENAME( @adjustedTableName )
	
	IF 	OBJECT_ID( @adjustedTableName ) IS NOT NULL AND 
		OBJECTPROPERTY( OBJECT_ID( @adjustedTableName ), N'IsUserTable' ) = 1 
	BEGIN
			SET @sql = N'DROP TABLE ' + @adjustedTableName
	
			EXEC @errorBuf = sp_executesql @sql
			
			IF COALESCE( NULLIF( @@ERROR, 0 ), @errorBuf ) <> 0 
			BEGIN
				RAISERROR( 'Error dropping table %s using dynamic SQL', 16, 1, @tableName )
				RETURN( 1 )
			END
	END
	
	SET @sql = N'SELECT * INTO ' + @tableName + N' FROM @UT_DD_DB_NAME@.' + @adjustedTableName
	EXEC @errorBuf = sp_executesql @sql
	
	IF COALESCE( NULLIF( @@ERROR, 0 ), @errorBuf ) <> 0 
	BEGIN
		RAISERROR( 'Error copying table %s from @UT_DD_DB_NAME@ using dynamic SQL', 16, 1, @adjustedTableName )
		RETURN( 1 )
	END
	
	RETURN( 0 )
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO



if exists (select * from dbo.sysobjects where id = object_id(N'[dbo].[utilsp_CopyMirrorLookupsFromDataDictionary]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [dbo].[utilsp_CopyMirrorLookupsFromDataDictionary]
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS ON 
GO

/*******************************************************************************
**	Name: 	utilsp_CopyMirrorLookupsFromDataDictionary
**
**	Desc:	Copies Class and MirrorIndexFieldLookup from DataDictionary 
**		to UNITTEST database
**
**	Input Parameters:
**		None.
**
**	Author:	Lohse
**	Date:	04/01/2011
********************************************************************************
**	Date Modified		Developer		Description
**	-------------		----------		------------------------
********************************************************************************/
CREATE PROCEDURE dbo.utilsp_CopyMirrorLookupsFromDataDictionary
AS

	DECLARE @errorBuf INT

	EXEC @errorBuf = dbo.utilsp_CopyTableFromDataDictionary '[dbo].[Class]'
	
	IF COALESCE( NULLIF( @@ERROR, 0 ), @errorBuf ) <> 0
	BEGIN
		RAISERROR( 'Error copying table [dbo].[Class] from the @UT_DD_DB_NAME@', 16, 1 )
		RETURN( 1 )
	END

	EXEC @errorBuf = dbo.utilsp_CopyTableFromDataDictionary '[dbo].[MirrorIndexFieldLookup]'
	
	IF COALESCE( NULLIF( @@ERROR, 0 ), @errorBuf ) <> 0
	BEGIN
		RAISERROR( 'Error copying table [dbo].[MirrorIndexFieldLookup] from the @UT_DD_DB_NAME@', 16, 1 )
		RETURN( 1 )
	END

	RETURN( 0 )
	


GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO



if exists (select * from dbo.sysobjects where id = object_id(N'[dbo].[utilsp_CreateEmptyMirrorDb]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [dbo].[utilsp_CreateEmptyMirrorDb]
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS ON 
GO

/*******************************************************************************
**	Name: 	utilsp_CreateEmptyMirrorDb
**
**	Desc:	Creates the Mirror database
**
**	Input Parameters:
**		None.
**
**	Author:	Lohse
**	Date:	04/01/2011
********************************************************************************
**	Date Modified		Developer		Description
**	-------------		----------		------------------------
********************************************************************************/
CREATE   PROCEDURE dbo.utilsp_CreateEmptyMirrorDb
AS

	DECLARE @mirrorDbName	SYSNAME
	DECLARE	@sql		NVARCHAR(4000)
	DECLARE @errorBuf	INT

    SET @mirrorDbName = dbo.utilfn_GetMirrorDbName() -- simply returns DB_NAME() + N'_MIRROR'

    IF @@ERROR <> 0 
    BEGIN
        RAISERROR( 'Error retrieving VBECS mirror DB name', 16, 1 )
        RETURN( 1 )
    END

   IF EXISTS (SELECT name FROM master.dbo.sysdatabases WHERE name = @mirrorDbName)
   BEGIN
	SET @sql = N' DROP DATABASE [' + @mirrorDbName + N']'
	EXEC(@sql)
   END

    IF @@ERROR <> 0 
    BEGIN
        RAISERROR( 'Error dropping existing VBECS mirror DB', 16, 1 )
        RETURN( 1 )
    END

-- Create the Mirrror Database Shell (using file locations of the main VBECS db)     
	DECLARE @DataFilePath NVARCHAR(1000)
	--using VBECS_VXXXX since it SHOULD exist..
	SELECT @DataFilePath = SUBSTRING(mf.physical_name, 1, CHARINDEX(N'@DB_NAME@', mf.physical_name) - 1)
	FROM master.sys.master_files mf
	JOIN master.sys.databases db ON mf.database_id = db.database_id AND db.name = '@DB_NAME@'
	WHERE mf.FILE_ID = 1

	--Get the path of the Log file
	DECLARE @LogFilePath NVARCHAR(1000)
	SELECT @LogFilePath = SUBSTRING(mf.physical_name, 1, CHARINDEX(N'@DB_NAME@', mf.physical_name) - 1)
	FROM master.sys.master_files mf
	JOIN master.sys.databases db ON mf.database_id = db.database_id AND db.name = '@DB_NAME@'
	WHERE mf.FILE_ID = 2

-- Create Mirror
	SET @sql = N'CREATE DATABASE ' + @mirrorDbName + ' ON  PRIMARY ' + 
			'( NAME = ''' + @mirrorDbName + '_DATA'', FILENAME = ''' + @DataFilePath + @mirrorDbName + '.MDF'', SIZE = 18304KB , MAXSIZE = UNLIMITED, FILEGROWTH = 10%)' +
		'LOG ON ' +
			'( NAME = ''' + @mirrorDbName + '_LOG'', FILENAME = ''' + @LogFilePath + @mirrorDbName + '.LDF'', SIZE = 13568KB , MAXSIZE = UNLIMITED, FILEGROWTH = 10%)'


    IF @@ERROR <> 0
    BEGIN
        RAISERROR( 'Error generating CREATE DATABASE dynamic SQL for the mirror DB', 16, 1 )
        RETURN( 1 )
    END

    EXEC( @sql )

    IF @@ERROR <> 0
    BEGIN
        RAISERROR( 'Error creating mirror DB with the dynamic SQL: %s', 16, 1, @sql )
        RETURN( 1 )
    END

    --sets the options in the database:
    EXEC @errorBuf = dbo.utilsp_SetNewMirrorDbOptions

    IF COALESCE(@errorBuf,@@ERROR) <> 0 
    BEGIN
        RAISERROR( 'Error setting new VBECS mirror DB options', 16, 1 )
        RETURN( 1 )
    END

    EXEC @errorBuf = dbo.utilsp_ExecuteSprocInMirrorDb 'sp_changedbowner ''sa'''

    IF COALESCE(@errorBuf,@@ERROR) <> 0 
    BEGIN
        RAISERROR( 'Error setting new VBECS mirror DB owner', 16, 1 )
        RETURN( 1 )
    END

-- function below simply returns "VbecsAppMirrorUser"
    SET @sql = 'sp_addrole ''VbecsAppMirrorUser'', ''dbo''' 

    EXEC @errorBuf = dbo.utilsp_ExecuteSprocInMirrorDb @sql

    IF COALESCE(@errorBuf,@@ERROR) <> 0 
    BEGIN
        RAISERROR( 'Error creating mirror user role in new mirror DB', 16, 1 )
        RETURN( 1 )
    END


    RETURN( 0 )
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO



if exists (select * from dbo.sysobjects where id = object_id(N'[dbo].[utilsp_CreateMirrorTableForTable]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [dbo].[utilsp_CreateMirrorTableForTable]
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS ON 
GO

/*******************************************************************************
**	Name: 	utilsp_CreateMirrorTableForTable
**
**	Desc:	Creates a Mirror table for the input table
**
**	Input Parameters:
**		tableName table in main database to create a Mirror copy of
**
**	Author:	Lohse
**	Date:	04/01/2011
********************************************************************************
**	Date Modified		Developer		Description
**	-------------		----------		------------------------
**	05/29/2012		Lohse			SQL 2008 upgrade - modified COALESCE
********************************************************************************/
CREATE  PROCEDURE dbo.utilsp_CreateMirrorTableForTable
(
    @tableName SYSNAME    
)
AS

	DECLARE @columnsList	NVARCHAR(4000)
	DECLARE @sql		NVARCHAR(4000)
	DECLARE @errorBuf	INT

	SELECT 		
		@columnsList = COALESCE( @columnsList + ', ', '' ) + C.column_name
	FROM   		
		INFORMATION_SCHEMA.TABLES T JOIN INFORMATION_SCHEMA.COLUMNS C ON T.table_name = C.table_name
	WHERE
		T.table_name = OBJECT_NAME( OBJECT_ID( @tableName ) )
	ORDER BY	
		C.ordinal_position

	IF @@ERROR <> 0 
	BEGIN
		RAISERROR( 'Error creating list of columns for the mirror table', 16, 1 )
		RETURN( 1 )
	END

	SET @sql = N'SELECT TOP 0 IDENTITY(INT,1,1) AS MirrorId, ' + REPLACE( @columnsList , 'RowVersion', '[RowVersion] = CAST(RowVersion AS BINARY(8))' ) + N', GETUTCDATE() AS MirrorDate ' + CHAR(13) + 
			   N'INTO ' + dbo.utilfn_GetMirrorDbName() + N'.dbo.' + dbo.utilfn_GetMirrorTableName( @tableName ) + CHAR(13) + 
			   N'FROM ' + @tableName

	IF @@ERROR <> 0 
	BEGIN
		RAISERROR( 'Error copmosing dynamic SQL for the mirror table', 16, 1 )
		RETURN( 1 )
	END

	--Hardcoded hack to get around Pending table snarfu	
	SET @sql = REPLACE(@sql, 'PendingBloodUnitTestId', 'cast(PendingBloodUnitTestId  as int) as PendingBloodUnitTestId')
	SET @sql = REPLACE(@sql, 'PendingSpecimenTestId', 'cast(PendingSpecimenTestId  as int) as PendingSpecimenTestId')
    
	EXEC( @sql )

	IF @@ERROR <> 0 
	BEGIN
		RAISERROR( 'Error creating a mirror table with the dynamic SQL: %s', 16, 1, @sql )
		RETURN( 1 )
	END
	
	SET @sql = N'GRANT INSERT, SELECT ON ' + dbo.utilfn_GetMirrorTableName( @tableName ) + N' TO VbecsAppMirrorUser'
	
	IF @@ERROR <> 0 
	BEGIN
		RAISERROR( 'Error creating dynamic SQL granting permissions on the mirror table', 16, 1 )
		RETURN( 1 )
	END


	EXEC @errorBuf = dbo.utilsp_ExecuteSqlInMirrorDb @sql
	
	IF COALESCE( @errorBuf,@@ERROR)  <> 0 
	BEGIN
		RAISERROR( 'Error granting permissions on the mirror table with dynamic SQL: %s', 16, 1, @sql )
		RETURN( 1 )
	END

    RETURN( 0 )
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO


if exists (select * from dbo.sysobjects where id = object_id(N'[dbo].[utilsp_CreateMirrorTableIndexesForTable]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [dbo].[utilsp_CreateMirrorTableIndexesForTable]
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS ON 
GO

/*******************************************************************************
**	Name: 	utilsp_CreateMirrorTableIndexesForTable
**
**	Desc:	Creates table indexes for Mirror table 
**
**	Input Parameters:
**		tableName table to create indexes for 
**
**	Author:	Lohse
**	Date:	04/01/2011
********************************************************************************
**	Date Modified		Developer		Description
**	-------------		----------		------------------------
********************************************************************************/
CREATE PROCEDURE dbo.utilsp_CreateMirrorTableIndexesForTable
(
    @tableName SYSNAME
)
AS

	DECLARE @mirrorTableName	SYSNAME
	DECLARE @sql		NVARCHAR(4000)
	DECLARE @errorBuf	INT
	DECLARE @origPrimaryKey	VARCHAR(255)
	DECLARE @fields2Index	VARCHAR(8000)

	SET @mirrorTableName = dbo.utilfn_GetMirrorTableName( @tableName )
	
	IF @@ERROR <> 0 
	BEGIN
		RAISERROR( 'Error getting mirror table name', 16, 1 )
		RETURN( 1 )
	END           
	
	SELECT 
		@origPrimaryKey = OrigPrimaryKey,
		@fields2Index = Fields2Index
	FROM
		dbo.MirrorIndexFieldLookup            
	WHERE
		TableName = @mirrorTableName

	IF @@ERROR <> 0 
	BEGIN
		RAISERROR( 'Error getting mirror table index information', 16, 1 )
		RETURN( 1 )
	END           
	
	IF @origPrimaryKey IS NULL AND @fields2Index IS NULL
	BEGIN
		SET @sql = N'ALTER TABLE dbo.' + @mirrorTableName + ' ADD CONSTRAINT PK_' + @mirrorTableName + ' PRIMARY KEY CLUSTERED (MirrorId) ON [PRIMARY]'
	END
	ELSE
	BEGIN
		SET @sql = N'ALTER TABLE dbo.' + @mirrorTableName + ' ADD CONSTRAINT PK_' + @mirrorTableName + ' PRIMARY KEY NONCLUSTERED (MirrorId) ON [PRIMARY]'
	END

	IF @@ERROR <> 0 
	BEGIN
		RAISERROR( 'Error preparing dynamic SQL for primary key on mirror table', 16, 1 )
		RETURN( 1 )
	END           
	
	EXEC @errorBuf = dbo.utilsp_ExecuteSqlInMirrorDb @sql
	
	IF COALESCE( @errorBuf,@@ERROR)  <> 0 
	BEGIN
		RAISERROR( 'Error creating primary key on mirrored table with dynamic SQL: %s', 16, 1, @sql )
		RETURN( 1 )
	END

	IF @origPrimaryKey IS NULL AND @fields2Index IS NULL RETURN( 0 )
	
	SET @sql = 	N'CREATE CLUSTERED  INDEX [IX_' + @mirrorTableName + '] ON dbo.' + @mirrorTableName + 
			N'( ' + REPLACE( @origPrimaryKey, '|', ',' ) + ' ) WITH  FILLFACTOR = 80 ON [PRIMARY]'
	
	IF @@ERROR <> 0 
	BEGIN
		RAISERROR( 'Error preparing dynamic SQL for clustered index on mirror table', 16, 1 )
		RETURN( 1 )
	END
	
	EXEC @errorBuf = dbo.utilsp_ExecuteSqlInMirrorDb @sql
	
	IF COALESCE( @errorBuf,@@ERROR)  <> 0 
	BEGIN
		RAISERROR( 'Error creating clustered index on mirrored table with dynamic SQL: %s', 16, 1, @sql )
		RETURN( 1 )
	END

	DECLARE @ColumnsToIndex TABLE( ColumnId INT IDENTITY(1,1), ColumnName SYSNAME )
	
	INSERT INTO @ColumnsToIndex
	(	
		ColumnName
	)
	SELECT	Item
	FROM
		dbo.fnSplit( @fields2Index, '|' )
	ORDER BY
		ItemId

	IF @@ERROR <> 0 
	BEGIN
		RAISERROR( 'Error getting a list of columns to index from a variable', 16, 1 )
		RETURN( 1 )
	END
	
	DECLARE @currentColumnId 	INT
	DECLARE @numberOfColumnsToIndex INT
	
	SELECT @numberOfColumnsToIndex = COUNT( 1 ) FROM @ColumnsToIndex
	
	SET @currentColumnId = 1

	WHILE @currentColumnId <= @numberOfColumnsToIndex
	BEGIN
		SELECT 
			@sql = 	N'CREATE NONCLUSTERED INDEX [IX_' + @mirrorTableName + '_' + ColumnName + N'] ' + 
				N'ON dbo.' + @mirrorTableName + N' ( ' + ColumnName + ' ) WITH  FILLFACTOR = 80 ON [PRIMARY] ' 
		FROM
			@ColumnsToIndex
		WHERE
			ColumnId = @currentColumnId
	
		IF @@ERROR <> 0 
		BEGIN
			RAISERROR( 'Error preparing create index dynamic SQL', 16, 1 )
			RETURN( 1 )
		END
		
		EXEC @errorBuf = dbo.utilsp_ExecuteSqlInMirrorDb @sql
		
		IF COALESCE( @errorBuf,@@ERROR)  <> 0 
		BEGIN
			RAISERROR( 'Error creating non-clustered index on mirrored table with dynamic SQL: %s', 16, 1, @sql )
			RETURN( 1 )
		END
	
		SET @currentColumnId = @currentColumnId + 1
	END

	SET @sql = 'sp_recompile ''' + @mirrorTableName + ''''
	
	EXEC @errorBuf = dbo.utilsp_ExecuteSprocInMirrorDb @sql
	
	IF COALESCE( @errorBuf,@@ERROR)  <> 0 
	BEGIN
		RAISERROR( 'Error marking mirror table for recompilation with dynamic SQL: %s', 16, 1, @sql )
		RETURN( 1 )
	END

    RETURN( 0 )
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[dbo].[utilsp_CreateMirrorViewForTable]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [dbo].[utilsp_CreateMirrorViewForTable]
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS ON 
GO

/*******************************************************************************
**	Name: 	utilsp_CreateMirrorViewForTable
**
**	Desc:	Creates mirror table view
**
**	Input Parameters:
**		tableName table to create view for 
**
**	Author:	Lohse
**	Date:	04/01/2011
********************************************************************************
**	Date Modified		Developer		Description
**	-------------		----------		------------------------
********************************************************************************/
CREATE    PROCEDURE dbo.utilsp_CreateMirrorViewForTable
(
    @tableName SYSNAME    
)
AS
	DECLARE @columnsList NVARCHAR(4000)
	
	SELECT 
		@columnsList = COALESCE( @columnsList + ', ', '' ) 
		+ CAST( REPLACE( C.[name], 'ROWVERSION', '[ROWVERSION]' ) AS NVARCHAR(255))
	FROM  
		sysobjects AS O	JOIN syscolumns AS C ON O.id = C.id
	WHERE 
		O.xtype = 'U'
	AND	O.type = 'U'
	AND	O.id = OBJECT_ID( @tableName )
	ORDER BY 
		colid

	IF @@ERROR <> 0 
	BEGIN
		RAISERROR( 'Error ocreating list of columns for the mirror view', 16, 1 )
		RETURN( 1 )
	END
	
	DECLARE @sql NVARCHAR(4000)

	SET @sql =	N'EXEC( ''' + CHAR(13) +
			N'CREATE VIEW ' + dbo.utilfn_GetMirrorViewName( @tableName ) + CHAR(13) +
			N'AS ' + CHAR(13) +
			N'SELECT Source=''''MIRROR'''', MirrorId, ' + @columnsList + N', MirrorDate' + CHAR(13) +
			N'FROM dbo.' + dbo.utilfn_GetMirrorTableName( @tableName ) + CHAR(13) +
			N'UNION ALL ' + CHAR(13) +
			N'SELECT Source=''''ACTIVE'''', 0, ' + @columnsList + + N', LastUpdateDate AS MirrorDate ' + CHAR(13) +
			N'FROM ' + DB_NAME() + '.dbo.' + OBJECT_NAME( OBJECT_ID( @tableName ) ) + ''')'

	IF @@ERROR <> 0 
	BEGIN
		RAISERROR( 'Error creating dynamic SQL for the mirror view', 16, 1 )
		RETURN( 1 )
	END

	DECLARE @errorBuf INT   
	
	EXEC @errorBuf = dbo.utilsp_ExecuteSqlInMirrorDb @sql
	
	IF COALESCE( @errorBuf,@@ERROR)  <> 0 
	BEGIN
		RAISERROR( 'Error creating a mirror view with the dynamic SQL: %s', 16, 1, @sql )
		RETURN( 1 )
	END
	
	SET @sql = N'GRANT SELECT ON ' + dbo.utilfn_GetMirrorViewName( @tableName ) + N' TO VbecsAppMirrorUser'
	
	IF @@ERROR <> 0 
	BEGIN
		RAISERROR( 'Error creating dynamic SQL granting permissions on the mirror view', 16, 1 )
		RETURN( 1 )
	END
	
	EXEC @errorBuf = dbo.utilsp_ExecuteSqlInMirrorDb @sql
	
	IF COALESCE( @errorBuf,@@ERROR)  <> 0 
	BEGIN
		RAISERROR( 'Error granting permissions on the mirror view with dynamic SQL: %s', 16, 1, @sql )
		RETURN( 1 )
	END

    RETURN( 0 )
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO



if exists (select * from dbo.sysobjects where id = object_id(N'[dbo].[utilsp_DropMirrorTriggerForTableIfExists]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [dbo].[utilsp_DropMirrorTriggerForTableIfExists]
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS ON 
GO

/*******************************************************************************
**	Name: 	utilsp_DropMirrorTriggerForTableIfExists
**
**	Desc:	Drop mirror trigger for individual table
**
**	Input Parameters:
**		tableName table to drop trigger for
**
**	Author:	Lohse
**	Date:	04/01/2011
********************************************************************************
**	Date Modified		Developer		Description
**	-------------		----------		------------------------
********************************************************************************/
CREATE  PROCEDURE dbo.utilsp_DropMirrorTriggerForTableIfExists
(
    @tableName SYSNAME
)
AS

	DECLARE @mirrorTriggerName SYSNAME
	
	SET @mirrorTriggerName = dbo.utilfn_GetMirrorTriggerName( @tableName )
	
	IF @@ERROR <> 0 
	BEGIN
		RAISERROR( 'Error creating mirror trigger name', 16, 1 )
		RETURN( 1 )
	END
	
	IF OBJECT_ID( @mirrorTriggerName ) IS NOT NULL AND OBJECTPROPERTY( OBJECT_ID( @mirrorTriggerName ), 'IsTrigger' ) = 1
	BEGIN  
		EXEC( N'DROP TRIGGER '+ @mirrorTriggerName )
	
		IF @@ERROR <> 0 
		BEGIN
			RAISERROR( 'Error dropping existing mirror trigger using dynamic SQL', 16, 1 )
			RETURN( 1 )
		END
	END
	
	IF @@ERROR <> 0 
	BEGIN
		RAISERROR( 'Error checking existence of a mirror trigger and remove it if needed', 16, 1 )
		RETURN( 1 )
	END

RETURN( 0 )
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO


if exists (select * from dbo.sysobjects where id = object_id(N'[dbo].[utilsp_CreateMirrorTriggerForTable]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [dbo].[utilsp_CreateMirrorTriggerForTable]
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS ON 
GO


/*******************************************************************************
**	Name: 	utilsp_CreateMirrorTriggerForTable
**
**	Desc:	Creates mirror trigger for individual table
**
**	Input Parameters:
**		tableName table to create trigger for
**
**	Author:	Lohse
**	Date:	04/01/2011
********************************************************************************
**	Date Modified		Developer		Description
**	-------------		----------		------------------------
********************************************************************************/
CREATE  PROCEDURE dbo.utilsp_CreateMirrorTriggerForTable
(
    @tableName SYSNAME
)
AS

	DECLARE @mirrorTriggerName SYSNAME
	
	SET @mirrorTriggerName = dbo.utilfn_GetMirrorTriggerName( @tableName )
	
	IF @@ERROR <> 0 
	BEGIN
		RAISERROR( 'Error creatig mirror trigger name', 16, 1 )
		RETURN( 1 )
	END
	
	DECLARE @sql NVARCHAR(4000)
	
	SET @sql =	N'CREATE TRIGGER '+ @mirrorTriggerName + CHAR(13) + 
			N'ON dbo.' + @tableName + CHAR(13) + 
			N'FOR UPDATE ' + CHAR(13) + 
			N'AS ' + CHAR(13) + 
			N'BEGIN' + CHAR(13) +
			N'	INSERT INTO ' + dbo.utilfn_GetMirrorDbName() + '.dbo.' + OBJECT_NAME( OBJECT_ID( @tableName ) ) + CHAR(13) +
			N'	SELECT *, GETUTCDATE()' + CHAR(13) +
			N'	FROM DELETED' + CHAR(13) +
			N'END' + CHAR(13)

	IF @@ERROR <> 0 
	BEGIN
		RAISERROR( 'Error creating dynamic SQL for the mirror trigger', 16, 1 )
		RETURN( 1 )
	END
	
	EXEC( @sql )
	
	IF @@ERROR <> 0 
	BEGIN
		RAISERROR( 'Error creating a mirror trigger using dynamic SQL: %s', 16, 1, @sql )
		RETURN( 1 )
	END
	
RETURN( 0 )
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO


if exists (select * from dbo.sysobjects where id = object_id(N'[dbo].[utilsp_SetUpMirroringForTable]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [dbo].[utilsp_SetUpMirroringForTable]
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS ON 
GO

/*******************************************************************************
**	Name: 	utilsp_SetUpMirroringForTable
**
**	Desc:	Flow to create mirror table with indexes, triggers and views
**
**	Input Parameters:
**		tableName table to create
**
**	Author:	Lohse
**	Date:	04/01/2011
********************************************************************************
**	Date Modified		Developer		Description
**	-------------		----------		------------------------
********************************************************************************/
CREATE    PROCEDURE dbo.utilsp_SetUpMirroringForTable
(
    @tableName SYSNAME
)
AS

	DECLARE @errorBuf INT
	
	EXEC @errorBuf = dbo.utilsp_CreateMirrorTableForTable @tableName
	
	IF COALESCE( @errorBuf,@@ERROR) <> 0
	BEGIN
		RAISERROR( 'Error creating new mirror table for table %s', 16, 1, @tableName )
		RETURN( 1 )
	END 
	
	EXEC @errorBuf = dbo.utilsp_CreateMirrorTableIndexesForTable @tableName
	
	IF COALESCE( @errorBuf,@@ERROR)  <> 0
	BEGIN
		RAISERROR( 'Error setting mirror indexes for mirror table of table %s', 16, 1, @tableName )
		RETURN( 1 )
	END 
	
	EXEC @errorBuf = dbo.utilsp_CreateMirrorViewForTable @tableName
	
	IF COALESCE( @errorBuf,@@ERROR)  <> 0
	BEGIN
		RAISERROR( 'Error creating new mirror view for table %s', 16, 1, @tableName )
		RETURN( 1 )
	END
	
	-- Preventive (most likely trigger exists already)
	EXEC @errorBuf = dbo.utilsp_DropMirrorTriggerForTableIfExists @tableName 
	
	IF COALESCE( @errorBuf,@@ERROR)  <> 0
	BEGIN
		RAISERROR( 'Error checking for existence of the mirror trigger for table %s and delete if needed', 16, 1, @tableName )
		RETURN( 1 )
	END
	
	EXEC @errorBuf = dbo.utilsp_CreateMirrorTriggerForTable @tableName
	
	IF COALESCE( @errorBuf,@@ERROR)  <> 0
	BEGIN
		RAISERROR( 'Error creating new mirror trigger for table %s', 16, 1, @tableName )
		RETURN( 1 )
	END    
	
RETURN( 0 )
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO



if exists (select * from dbo.sysobjects where id = object_id(N'[dbo].[utilsp_SetupMirroringForAllTables]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [dbo].[utilsp_SetupMirroringForAllTables]
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS ON 
GO

/*******************************************************************************
**	Name: 	utilsp_SetupMirroringForAllTables
**
**	Desc:	Uses Class table from VbecsDataDictionary database to determine
**		which tables need mirroring.
**
**	Input Parameters:
**		None.
**
**	Author:	Lohse
**	Date:	04/01/2011
********************************************************************************
**	Date Modified		Developer		Description
**	-------------		----------		------------------------
********************************************************************************/
CREATE PROCEDURE dbo.utilsp_SetupMirroringForAllTables
AS    
	DECLARE @MirroredTables TABLE( TableId INT IDENTITY(1,1), TableName SYSNAME )
	
	INSERT INTO @MirroredTables
	( 
		TableName 
	)
	SELECT
		[ClassName]
	FROM
		[Class]
	WHERE
		MirrorFlag = 1

	IF @@ERROR <> 0 
	BEGIN
		RAISERROR( 'Error creating list of mirrored table names into a table variable', 16, 1 )
		RETURN( 1 )
	END

	DECLARE @errorBuf	INT
	DECLARE @tableName	SYSNAME
	DECLARE @currentTableId	INT
	DECLARE @numberOfTablesToProcess	INT

	SET @currentTableId = 1
	
	SELECT @numberOfTablesToProcess = COUNT(1) FROM @MirroredTables
	
	IF @@ERROR <> 0 
	BEGIN
		RAISERROR( 'Error populating the @numberOfTablesToProcess variable', 16, 1 )
		RETURN( 1 )
	END
	
	WHILE @currentTableId <= @numberOfTablesToProcess
	BEGIN
		SELECT 
			@tableName = TableName
		FROM
			@MirroredTables
		WHERE
			TableId = @currentTableId
		
		IF @@ERROR <> 0 
		BEGIN
			RAISERROR( 'Error getting next table from @MirroredTables', 16, 1 )
			RETURN( 1 )
		END
	
	        EXEC @errorBuf = dbo.utilsp_SetUpMirroringForTable @tableName
	
	        IF COALESCE( @errorBuf,@@ERROR) <> 0
		        BEGIN
		            RAISERROR( 'Error setting mirroring for table %s', 16, 1, @tableName )
		            RETURN( 1 ) 
		        END
	
	        SET @currentTableId = @currentTableId + 1
	END

RETURN( 0 )
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO



if exists (select * from dbo.sysobjects where id = object_id(N'[dbo].[utilsp_BuildMirrorDb]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [dbo].[utilsp_BuildMirrorDb]
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS ON 
GO

/*******************************************************************************
**	Name: 	utilsp_BuildMirrorDb
**
**	Desc:	Flow to control the creation of the Mirror database
**
**	Input Parameters:
**		None.
**
**	Author:	Lohse
**	Date:	04/01/2011
********************************************************************************
**	Date Modified		Developer		Description
**	-------------		----------		------------------------
********************************************************************************/
CREATE PROCEDURE dbo.utilsp_BuildMirrorDb
AS
    SET NOCOUNT ON

	DECLARE @errorBuf	INT
	DECLARE @returnCode	INT

	SET @returnCode = 1 -- fail by default
	
	EXEC @errorBuf = dbo.utilsp_CreateEmptyMirrorDb
	
	IF COALESCE(@errorBuf,@@ERROR) <> 0 
	BEGIN
		RAISERROR( 'Error dropping and create mirror DB', 16, 1 )
		GOTO CLEANUP_AND_EXIT
	END
	
	EXEC @errorBuf = dbo.utilsp_SetupMirroringForAllTables
	
	IF COALESCE(@errorBuf,@@ERROR) <> 0 
	BEGIN
		RAISERROR( 'Error setting mirroring for all appropriate tables', 16, 1 )
		GOTO CLEANUP_AND_EXIT
	END
	
	EXEC @errorBuf = dbo.utilsp_UpdateMirrorDbNameInRoutines
	
	IF COALESCE(@errorBuf,@@ERROR) <> 0 
	BEGIN
		RAISERROR( 'Error updating mirror DB name in primary DB routines', 16, 1 )
		GOTO CLEANUP_AND_EXIT
	END       
	
	SET @returnCode = 0

CLEANUP_AND_EXIT:

SET NOCOUNT OFF

RETURN( @returnCode )
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO


if exists (select * from dbo.sysobjects where id = object_id(N'[dbo].[utilsp_CopyReferenceTableDataFromDesign]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [dbo].[utilsp_CopyReferenceTableDataFromDesign]
GO
SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS ON 
GO

/*******************************************************************************
**	Name: 	utilsp_CopyReferenceTableDataFromDesign
**
**	Desc:	Copies all tables (containing data) from the DESIGN database to
**		the current database. Assumes tables are already empty.
**
**	Input Parameters:
**		None.
**
**	Author:	Lohse
**	Date:	04/01/2011
********************************************************************************
**	Date Modified		Developer		Description
**	-------------		----------		------------------------
**	06/22/2012		Lohse			Fixing IDENTITY workings for SQL2008
********************************************************************************/
CREATE PROCEDURE dbo.utilsp_CopyReferenceTableDataFromDesign 
AS
        DECLARE @errorBuf INT

        EXEC @errorBuf = dbo.utilsp_EnableAllReferentialConstraints 0
        
        IF COALESCE( NULLIF( @@ERROR, 0 ), @errorBuf ) <> 0
        BEGIN
                RAISERROR( 'Error disabling all table constraints', 16, 1 )
                RETURN( 1 )
        END
        
        
        CREATE TABLE #tmpRefresh 
        (
                Counter                INT IDENTITY (1, 1),
                TableName        VARCHAR(100),
                ColumnList        VARCHAR(4000),
                IdentityOnSql        VARCHAR(400),
                IdentityOffSql        VARCHAR(400)
        )
        
        INSERT INTO #tmpRefresh 
        (
                TableName,
                ColumnList,
                IdentityOnSql,
				IdentityOffSql
        )
        SELECT 
                so.name,
                dbo.utilfn_GetColumnListForInsert(so.name),
                NULL,
                ';'
        FROM 
                @UT_DESIGN_DB_NAME@.dbo.sysobjects so JOIN @UT_DESIGN_DB_NAME@.dbo.sysindexes si ON so.id = si.id
        WHERE
                so.xtype = 'U'
        AND        si.indid <= 1
        AND        si.rows > 0
        AND        so.name NOT IN ('dtproperties')
        AND        so.name NOT LIKE 'tmpl%' 


        UPDATE #tmpRefresh
        SET IdentityOnSql = 'SET IDENTITY_INSERT ' + TableName + ' ON ',
        IdentityOffSql = ';SET IDENTITY_INSERT ' + TableName + ' OFF ;'
        WHERE TableName IN (SELECT DISTINCT so.name FROM @UT_DESIGN_DB_NAME@.sys.sysobjects so 
                                        JOIN @UT_DESIGN_DB_NAME@.sys.identity_columns sc ON so.id = sc.object_id)

------------------------------------------------------------------------------------------------------
        DECLARE @tableId         INT
        DECLARE @tableCount         INT
        DECLARE @sql                 NVARCHAR(4000)


        SELECT @tableCount = COUNT(1) FROM #tmpRefresh
        SET @tableId = 1

        WHILE @tableId <= @tableCount
        BEGIN
                SELECT 
                        @sql = CASE WHEN IdentityOnSql IS NOT NULL THEN IdentityOnSql + N';' ELSE ' ' END + N'INSERT INTO ' + TableName + '(' + ColumnList + ') SELECT ' + ColumnList + ' FROM @UT_DESIGN_DB_NAME@.dbo.' + TableName + IdentityOffSql
                FROM
                        #tmpRefresh
                WHERE
                        Counter = @tableId

                IF @@ERROR <> 0 
                BEGIN
                        RAISERROR( 'Error fetching from #tmpRefresh table', 16, 1 )
                        RETURN( 1 )
                END
            
                EXEC @errorBuf = sp_executesql @sql 

                IF COALESCE( NULLIF( @@ERROR, 0 ), @errorBuf ) <> 0
                BEGIN
                            RAISERROR( 'Error executing dymaic sql: %s', 16, 1, @sql )
                        RETURN( 1 )
                END

                SET @tableId = @tableId + 1
        END  

------------------------------------------------------------------------------------------------------


        EXEC @errorBuf = dbo.utilsp_EnableAllReferentialConstraints 1
        
        IF COALESCE( NULLIF( @@ERROR, 0 ), @errorBuf ) <> 0
        BEGIN
                RAISERROR( 'Error enabling all table constraints', 16, 1 )
                RETURN( 1 )
        END

RETURN( 0 )

GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO


if exists (select * from dbo.sysobjects where id = object_id(N'[dbo].[utilsp_RegenerateMirrorDbInDevelopment]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [dbo].[utilsp_RegenerateMirrorDbInDevelopment]
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS ON 
GO

/*******************************************************************************
**	Name: 	utilsp_RegenerateMirrorDbInDevelopment
**
**	Desc:	Master sproc to drop and create a new blank mirror database
**
**	Input Parameters:
**		None.
**
**	Author:	Lohse
**	Date:	04/01/2011
********************************************************************************
**	Date Modified		Developer		Description
**	-------------		----------		------------------------
********************************************************************************/
CREATE    PROCEDURE dbo.utilsp_RegenerateMirrorDbInDevelopment
AS

	DECLARE @errorBuf	INT
	DECLARE @mirrorDbName	SYSNAME       

	SET @mirrorDbName = dbo.utilfn_GetMirrorDbName()
	
	IF @@ERROR <> 0 
	BEGIN
		RAISERROR( 'Error getting VBECS mirror DB name', 16, 1 )
		RETURN( 1 )
	END            
	
	IF EXISTS( SELECT 1 FROM master..sysdatabases WHERE [name] = @mirrorDbName )
	BEGIN
		EXEC @errorBuf = dbo.utilsp_KillDBUsers @mirrorDbName
	
		IF COALESCE( @errorBuf,@@ERROR ) <> 0 
		BEGIN
			RAISERROR( 'Error killing active DB users in VBECS mirror DB', 16, 1 )
			RETURN( 1 )
		END
	
		DECLARE @sql NVARCHAR(4000)
	
		SET @sql = N'DROP DATABASE ' + @mirrorDbName
	
		EXEC( @sql )
	
		IF @@ERROR <> 0 
		BEGIN
			RAISERROR( 'Error dropping VBECS mirror DB', 16, 1 )
			RETURN( 1 )
		END

		-- Clean the proc cache so that no procs will point to the wrong mirror
		DBCC FREEPROCCACHE
		
		IF @@ERROR <> 0 
		BEGIN
			RAISERROR( 'Error with FREEPROCCACHE after dropping the mirror DB', 16, 1 )
			RETURN( 1 )
		END    
	END    

	EXEC @errorBuf = dbo.utilsp_CopyMirrorLookupsFromDataDictionary
	
	IF COALESCE( @errorBuf,@@ERROR ) <> 0 
	BEGIN
		RAISERROR( 'Error copying mirror lookup tables from the VBECS_DataDictionary DB', 16, 1 )
		RETURN( 1 )
	END
	
	EXEC @errorBuf = dbo.utilsp_BuildMirrorDb
	
	IF COALESCE( @errorBuf,@@ERROR )<> 0 
	BEGIN
		RAISERROR( 'Error rebuilding mirror DB', 16, 1 )
		RETURN( 1 )
	END
	
RETURN( 0 )
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO
